"""
$Id: FSSTool.py,v 1.9 2005/03/30 13:18:03 clebeaupin Exp $
"""

__author__  = ''
__docformat__ = 'restructuredtext'

# Python imports
import os
import re
import random
import time
import Globals

# Zope imports
from Globals import package_home
from Globals import InitializeClass
from AccessControl import ClassSecurityInfo
from OFS.SimpleItem import SimpleItem
from OFS.PropertyManager import PropertyManager
from DateTime import DateTime
from ZPublisher.Iterators import IStreamIterator

# CMF imports
from Products.CMFCore.utils import UniqueObject, getToolByName
from Products.CMFCore.ActionProviderBase import ActionProviderBase
from Products.CMFCore import CMFCorePermissions
from Products.CMFCore.ActionInformation import ActionInformation
from Products.CMFCore.Expression import Expression

# Products imports
from FileUtils import move_file
from Products.FileSystemStorage.FileSystemStorage import FileSystemStorage

class FSSTool(PropertyManager, UniqueObject, SimpleItem, ActionProviderBase):
    """Tool for FileSystem storage"""

    plone_tool = 1
    id = 'portal_fss'
    title = 'FSSTool'
    storage_path = ''
    backup_path = ''
    rdf_enabled = False
    rdf_script = ''
    meta_type = 'FSSTool'
    
    _properties=(
        {'id':'title', 'type': 'string', 'mode':'w'},
        {'id':'storage_path', 'type': 'string', 'mode':'w'},
        {'id':'backup_path', 'type': 'string', 'mode':'w'},
        {'id':'rdf_enabled', 'type': 'boolean', 'mode':'w'},
        {'id':'rdf_script', 'type': 'string', 'mode':'w'},
        )
        
        
    _actions = ()
            
    manage_options = (ActionProviderBase.manage_options + PropertyManager.manage_options + SimpleItem.manage_options)

    security = ClassSecurityInfo()

    security.declarePrivate('manage_afterAdd')
    def manage_afterAdd(self, item, container):
        self.initProperties()

    security.declareProtected(CMFCorePermissions.ManagePortal, 'initProperties')
    def initProperties(self):
        """Init properties"""
        
        default_path = os.path.join(Globals.INSTANCE_HOME, 'var')
        self.storage_path = default_path
        self.backup_path = default_path
    
    security.declareProtected(CMFCorePermissions.View, 'isRDFEnabled')
    def isRDFEnabled(self):
        """Returns true if RDF is automaticaly generated when file added"""
    
        return self.rdf_enabled
    
    security.declareProtected(CMFCorePermissions.ManagePortal, 'enableRDF')
    def enableRDF(self, enabled):
        """Enable rdf or not"""
    
        if enabled:
            self.rdf_enabled = True
        else:
            self.rdf_enabled = False
    
    security.declareProtected(CMFCorePermissions.View, 'getRDFScript')
    def getRDFScript(self):
        """Returns rdf script used to generate RDF on files"""
    
        return self.rdf_script
    
    security.declareProtected(CMFCorePermissions.ManagePortal, 'setRDFScript')
    def setRDFScript(self, rdf_script):
        """Set rdf script used to generate RDF on files"""
    
        self.rdf_script = rdf_script
    
    security.declareProtected(CMFCorePermissions.View, 'getStoragePath')
    def getStoragePath(self):
        """storage_path accessor"""
    
        return self.storage_path
    
    security.declareProtected(CMFCorePermissions.View, 'getBackupPath')
    def getBackupPath(self):
        """backup_path accessor"""
        
        return self.backup_path
        
    security.declareProtected(CMFCorePermissions.ManagePortal, 'setStoragePath')
    def setStoragePath(self, storage_path):
        """storage_path mutator"""
        
        self.storage_path = storage_path
    
    security.declareProtected(CMFCorePermissions.ManagePortal, 'setBackupPath')
    def setBackupPath(self, backup_path):
        """backup_path mutator"""
        
        self.backup_path = backup_path
    
    security.declareProtected(CMFCorePermissions.ManagePortal, 'getFSItems')
    def getFSItems(self, path, extension=''):
        """Returns list of items in filesystem.
        Item is a dictionnary :
        uid -> uid of linked content
        field -> name of field
        modified -> DateTime of last modification
        zope_path -> Zope path
        """
        
        items = []
        uid_paths = self.getContentUIDPaths()
        reg = r'^(?P<uid>.{32})_(?P<field>[^.]*)'
        
        if extension:
            reg += extension

        reg += '$'            
        re_item = re.compile(reg)
        
        # Walk into filesystem
        for root, dirs, files in os.walk(path):
            if root == path:
                # Loop on files
                for item in files:
                    match = re_item.match(item)
                    
                    if match is not None:
                        uid = match.group('uid')
                        field = match.group('field')
                        file_path = os.path.join(root, item)
                        zope_path = uid_paths.get(uid, None)
                        modified = os.path.getmtime(file_path)
                        size = os.path.getsize(file_path)

                        # Store item
                        item = {}
                        item['uid'] = uid
                        item['field'] = field
                        item['modified'] = DateTime(modified)
                        item['zope_path'] = zope_path
                        item['size'] = size
                        items.append(item)
                        
        return tuple(items)
    
    security.declareProtected(CMFCorePermissions.ManagePortal, 'getStorageItems')
    def getStorageItems(self):
        """Returns list of items in storage path"""
        
        path = self.getStoragePath()
        return self.getFSItems(path)
    
    security.declareProtected(CMFCorePermissions.ManagePortal, 'getBackupFSSItems')
    def getBackupItems(self):
        """Returns list of items in backup path"""
        
        path = self.getBackupPath()
        return self.getFSItems(path, '.bak')
        
        
    security.declareProtected(CMFCorePermissions.ManagePortal, 'getFSStats')
    def getFSStats(self):
        """
        Returns stats on FileSystem storage
        valid_files_count -> Count of valid files
        not_valid_files_count -> Count of not valid files
        valid_backups_count -> Count of valid backups
        not_valid_backups_count -> Count of not valid backups
        """
        
        storage_items = self.getStorageItems()
        backup_items = self.getBackupItems()
        
        valid_files = [x for x in storage_items if x['zope_path'] is not None]
        not_valid_files = [x for x in storage_items if x['zope_path'] is None]
        valid_backups = [x for x in backup_items if x['zope_path'] is None]
        not_valid_backups = [x for x in backup_items if x['zope_path'] is not None]
        
        
        # Sort valid files by size
        def cmp_size(a, b):
              return cmp(a['size'], b['size'])
              
        valid_files.sort(cmp_size)

        # Size in octets
        total_size = 0
        largest_size = 0
        smallest_size = 0
        average_size = 0
        
        for x in valid_files:
            total_size += x['size']
            
        if len(valid_files) > 0:
            largest_size = valid_files[-1]['size']
            smallest_size = valid_files[0]['size']
            average_size = int(total_size / len(valid_files))
        
        stats = {
          'valid_files_count' : len(valid_files),
          'not_valid_files_count' : len(not_valid_files),
          'valid_backups_count' : len(valid_backups),
          'not_valid_backups_count' : len(not_valid_backups),
          'total_size' : total_size,
          'largest_size': largest_size,
          'smallest_size' : smallest_size,
          'average_size' : average_size,
          }
        
        return stats
    
    security.declareProtected(CMFCorePermissions.ManagePortal, 'updateFSS')
    def updateFSS(self):
        """
        Update FileSystem storage
        """
        
        storage_path = self.getStoragePath()
        backup_path = self.getBackupPath()
        storage_items = self.getStorageItems()
        backup_items = self.getBackupItems()
        
        not_valid_files = tuple([x for x in storage_items if x['zope_path'] is None])
        not_valid_backups = tuple([x for x in backup_items if x['zope_path'] is not None])
        
        # Move not valid files in backup
        for item in not_valid_files:
            src_filename = '%s_%s' % (item['uid'], item['field'])
            dst_filename = '%s.bak' % src_filename
            src_path = os.path.join(storage_path, src_filename)
            dst_path = os.path.join(backup_path, dst_filename)
            move_file(src_path, dst_path)
            
        # Move not valid backups in file storage
        for item in not_valid_backups:
            dst_filename = '%s_%s' % (item['uid'], item['field'])
            src_filename = '%s.bak' % dst_filename
            dst_path = os.path.join(storage_path, dst_filename)
            src_path = os.path.join(backup_path, src_filename)
            move_file(src_path, dst_path)
        
    security.declareProtected(CMFCorePermissions.ManagePortal, 'removeBackups')
    def removeBackups(self, max_days):
        """
        Remove backups older than specified days
        """
        
        backup_path = self.getBackupPath()
        backup_items = self.getBackupItems()
        valid_backups = [x for x in backup_items if x['zope_path'] is None]
        current_time = time.time()
        
        for item in valid_backups:
            one_day = 86400 # One day 86400 seconds
            modified = item['modified']
            seconds = int(current_time) - int(modified.timeTime())
            days = int(seconds/one_day)
            
            if days >= max_days:
                filename = '%s_%s.bak' % (item['uid'], item['field'])
                path = os.path.join(backup_path, filename)
                os.remove(path)
                
    security.declareProtected(CMFCorePermissions.ManagePortal, 'getContentUIDPaths')
    def getContentUIDPaths(self):
        """Returns a dictionnary. For all uid give the path"""
        
        utool = getToolByName(self, 'uid_catalog')
        brains = utool()
        return dict([(x['UID'], x.getPath()) for x in brains])
    
    security.declareProtected(CMFCorePermissions.ManagePortal, 'updateRDF')
    def updateRDF(self):
        """Add rdf files to fss files"""
        
        storage_items = self.getStorageItems()
        for item in storage_items:
            obj_path = item['zope_path']
            if obj_path is not None:
                obj = self.restrictedTraverse(obj_path)
                field_name = item['field']
                field = obj.getField(field_name)
                if field is None:
                    continue
                storage = field.getStorage(obj)
                if not isinstance(storage, FileSystemStorage):
                    continue
                storage.updateRDF(name=field_name, instance=obj, rdf_script=self.getRDFScript())
    
InitializeClass(FSSTool)
